document.addEventListener('DOMContentLoaded', function() {
    // Handle mobile menu toggle
    const menuIcon = document.querySelector('.menu-icon');
    const navLinks = document.querySelector('.nav-links');
    
    if (menuIcon) {
        menuIcon.addEventListener('click', function() {
            navLinks.classList.toggle('active');
        });
    }
    
    // Search functionality
    const searchForm = document.querySelector('.search-box');
    const searchInput = searchForm?.querySelector('input[name="q"]');
    
    if (searchForm && searchInput) {
        searchForm.addEventListener('submit', function(e) {
            const searchTerm = searchInput.value.trim();
            if (!searchTerm) {
                e.preventDefault();
            }
        });
    }
    
    // Theme toggle functionality
    const themeToggle = document.getElementById('theme-toggle');
    
    // Check for saved theme preference or use device preference
    const savedTheme = localStorage.getItem('theme');
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    
    // Set initial theme
    if (savedTheme) {
        document.documentElement.setAttribute('data-theme', savedTheme);
        if (savedTheme === 'light') {
            document.querySelector('.dark-icon').style.display = 'none';
            document.querySelector('.light-icon').style.display = 'block';
        }
    } else if (!prefersDark) {
        document.documentElement.setAttribute('data-theme', 'light');
        document.querySelector('.dark-icon').style.display = 'none';
        document.querySelector('.light-icon').style.display = 'block';
    }
    
    // Toggle theme on button click
    if (themeToggle) {
        themeToggle.addEventListener('click', function() {
            const currentTheme = document.documentElement.getAttribute('data-theme') || 'dark';
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            
            // Update theme attribute
            document.documentElement.setAttribute('data-theme', newTheme);
            
            // Update icon display
            const darkIcon = document.querySelector('.dark-icon');
            const lightIcon = document.querySelector('.light-icon');
            
            if (newTheme === 'light') {
                darkIcon.style.display = 'none';
                lightIcon.style.display = 'block';
            } else {
                darkIcon.style.display = 'block';
                lightIcon.style.display = 'none';
            }
            
            // Save preference to localStorage
            localStorage.setItem('theme', newTheme);
        });
    }
    
    // Add active class to current page in navigation
    const currentLocation = window.location.pathname;
    const navItems = document.querySelectorAll('.nav-links a, .menu-list a');
    
    navItems.forEach(item => {
        const href = item.getAttribute('href');
        if (currentLocation.includes(href) && href !== '#') {
            item.classList.add('active');
        }
    });
    
    // Server status check simulation
    function checkServerStatus() {
        // In a real application, this would be an AJAX call to the server
        // For demo, we'll just randomly change status sometimes
        
        const statusIndicators = document.querySelectorAll('.status-indicator');
        const statusTexts = document.querySelectorAll('.status-text');
        
        // We'll have the server mostly online, occasionally offline
        const isOnline = Math.random() > 0.1; // 90% chance to be online
        
        statusIndicators.forEach(indicator => {
            if (isOnline) {
                indicator.classList.remove('offline');
                indicator.classList.add('online');
            } else {
                indicator.classList.remove('online');
                indicator.classList.add('offline');
            }
        });
        
        statusTexts.forEach(text => {
            text.textContent = isOnline ? 'Server Online' : 'Server Offline';
        });
    }
    
    // Check server status periodically (every 30 seconds)
    setInterval(checkServerStatus, 30000);
    
    // Handle news item expansion
    const expandButtons = document.querySelectorAll('.expand-btn');
    
    expandButtons.forEach(button => {
        button.addEventListener('click', function() {
            const newsItem = this.closest('.news-item');
            const newsText = newsItem.querySelector('.news-text');
            
            if (this.textContent === '+') {
                this.textContent = '-';
                newsText.style.maxHeight = newsText.scrollHeight + 'px';
            } else {
                this.textContent = '+';
                newsText.style.maxHeight = null;
            }
        });
    });
    
    // Language selector functionality
    const languageSelector = document.querySelector('.language-selector');
    const currentFlag = document.getElementById('current-flag');
    
    if (languageSelector) {
        languageSelector.addEventListener('click', function(e) {
            this.classList.toggle('active');
        });
        
        // Close language selector when clicking outside
        document.addEventListener('click', function(e) {
            if (!languageSelector.contains(e.target)) {
                languageSelector.classList.remove('active');
            }
        });
    }
    
    // Initialize language from localStorage or default to English
    const currentLang = localStorage.getItem('language') || 'en';
    changeLanguage(currentLang);
});

// Language change function
function changeLanguage(lang) {
    // Update current flag
    const currentFlag = document.getElementById('current-flag');
    currentFlag.src = `images/flags/${lang}.png`;
    
    // Save language preference
    localStorage.setItem('language', lang);
    
    // Update text content based on selected language
    const elements = document.querySelectorAll('[data-i18n]');
    elements.forEach(element => {
        const key = element.getAttribute('data-i18n');
        if (translations[lang] && translations[lang][key]) {
            element.textContent = translations[lang][key];
        }
    });
    
    // Close language selector dropdown
    document.querySelector('.language-selector').classList.remove('active');
} 