document.addEventListener('DOMContentLoaded', function() {
    // Handle mobile menu toggle
    const menuToggle = document.getElementById('menu-toggle');
    const navLinks = document.querySelector('.nav-links');
    
    if (menuToggle) {
        menuToggle.addEventListener('change', function() {
            if (this.checked) {
                navLinks.classList.add('active');
            } else {
                navLinks.classList.remove('active');
            }
        });
    }
    
    // Theme toggle functionality
    const themeToggle = document.getElementById('theme-toggle');
    
    // Check for saved theme preference or use device preference
    const savedTheme = localStorage.getItem('theme');
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    
    // Set initial theme
    if (savedTheme) {
        document.documentElement.setAttribute('data-theme', savedTheme);
        if (savedTheme === 'light') {
            document.querySelector('.dark-icon').style.display = 'none';
            document.querySelector('.light-icon').style.display = 'block';
        }
    } else if (!prefersDark) {
        document.documentElement.setAttribute('data-theme', 'light');
        document.querySelector('.dark-icon').style.display = 'none';
        document.querySelector('.light-icon').style.display = 'block';
    }
    
    // Toggle theme on button click
    if (themeToggle) {
        themeToggle.addEventListener('click', function() {
            const currentTheme = document.documentElement.getAttribute('data-theme') || 'dark';
            const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
            
            // Update theme attribute
            document.documentElement.setAttribute('data-theme', newTheme);
            
            // Update icon display
            const darkIcon = document.querySelector('.dark-icon');
            const lightIcon = document.querySelector('.light-icon');
            
            if (newTheme === 'light') {
                darkIcon.style.display = 'none';
                lightIcon.style.display = 'block';
            } else {
                darkIcon.style.display = 'block';
                lightIcon.style.display = 'none';
            }
            
            // Save preference to localStorage
            localStorage.setItem('theme', newTheme);
        });
    }
    
    // Add active class to current page in navigation
    const currentLocation = window.location.pathname;
    const navItems = document.querySelectorAll('.nav-links a');
    
    navItems.forEach(item => {
        const href = item.getAttribute('href');
        if (currentLocation.includes(href) && href !== '#') {
            item.classList.add('active');
        }
    });
    
    // Server status check simulation
    function checkServerStatus() {
        // In a real application, this would be an AJAX call to the server
        // For demo, we'll just randomly change status sometimes
        
        const statusIndicators = document.querySelectorAll('.status-indicator');
        const statusTexts = document.querySelectorAll('.status-text');
        
        // We'll have the server mostly online, occasionally offline
        const isOnline = Math.random() > 0.1; // 90% chance to be online
        
        statusIndicators.forEach(indicator => {
            if (isOnline) {
                indicator.classList.remove('offline');
                indicator.classList.add('online');
            } else {
                indicator.classList.remove('online');
                indicator.classList.add('offline');
            }
        });
        
        statusTexts.forEach(text => {
            text.textContent = isOnline ? 'Server Online' : 'Server Offline';
        });
    }
    
    // Check server status periodically (every 30 seconds)
    // In a real application, you might want to use WebSockets for this
    setInterval(checkServerStatus, 30000);

    // News ticker functionality
    document.querySelectorAll('.changelog_trigger').forEach(trigger => {
        trigger.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('targetid');
            
            document.querySelector('.minus' + targetId).style.display = 
                document.querySelector('.minus' + targetId).style.display === 'none' ? 'inline-block' : 'none';
            
            document.querySelector('.plus' + targetId).style.display = 
                document.querySelector('.plus' + targetId).style.display === 'none' ? 'inline-block' : 'none';
            
            const bigChangelog = document.querySelector('.changelog_big' + targetId);
            const smallChangelog = document.querySelector('.changelog_small' + targetId);
            
            bigChangelog.classList.toggle('display-inline');
            smallChangelog.classList.toggle('display-none');
        });
    });

    // Character search form validation
    const searchForm = document.querySelector('form.search-character');
    if (searchForm) {
        searchForm.addEventListener('submit', function(e) {
            const input = this.querySelector('input[name="name"]').value;
            if (input.length < 3) {
                e.preventDefault();
                alert('Please enter at least 3 characters.');
            }
        });
    }

    // Market functionality
    function initializeMarket() {
        // Copy item code to clipboard
        const copyButtons = document.querySelectorAll('.copy-btn');
        copyButtons.forEach(button => {
            button.addEventListener('click', function() {
                const code = this.getAttribute('data-code');
                navigator.clipboard.writeText(code).then(() => {
                    const originalText = this.textContent;
                    this.textContent = 'Copied!';
                    setTimeout(() => {
                        this.textContent = originalText;
                    }, 2000);
                });
            });
        });
    }

    // Gift system functionality
    function initializeGiftSystem() {
        // Filter buttons
        const filterButtons = document.querySelectorAll('.filter-btn');
        const giftItems = document.querySelectorAll('.gift-item');
        
        filterButtons.forEach(button => {
            button.addEventListener('click', function() {
                const category = this.getAttribute('data-category');
                
                // Update active button
                filterButtons.forEach(btn => btn.classList.remove('active'));
                this.classList.add('active');
                
                // Filter items
                giftItems.forEach(item => {
                    const itemCategory = item.getAttribute('data-category');
                    if (category === 'all' || itemCategory === category) {
                        item.style.display = 'block';
                    } else {
                        item.style.display = 'none';
                    }
                });
            });
        });
        
        // Gift purchase modal
        const giftButtons = document.querySelectorAll('.gift-btn');
        const giftModal = document.querySelector('.gift-modal');
        const closeModal = document.querySelector('.gift-modal-close');
        
        giftButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const giftId = this.getAttribute('data-gift-id');
                const giftName = this.getAttribute('data-gift-name');
                const giftPrice = this.getAttribute('data-gift-price');
                
                // Update modal content
                document.querySelector('.gift-modal-title').textContent = `Purchase ${giftName}`;
                document.querySelector('.gift-price-display').textContent = `${giftPrice} points`;
                document.querySelector('#gift-id').value = giftId;
                
                // Show modal
                giftModal.style.display = 'flex';
            });
        });
        
        if (closeModal) {
            closeModal.addEventListener('click', () => {
                giftModal.style.display = 'none';
            });
        }
        
        // Close modal on outside click
        window.addEventListener('click', (e) => {
            if (e.target === giftModal) {
                giftModal.style.display = 'none';
            }
        });
    }

    // Points system functionality
    function initializePointsSystem() {
        const buyButtons = document.querySelectorAll('.buy-btn');
        const pointsModal = document.querySelector('.payment-modal');
        const closeModal = document.querySelector('.payment-modal-close');
        
        buyButtons.forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                const packageId = this.getAttribute('data-package-id');
                const points = this.getAttribute('data-points');
                const price = this.getAttribute('data-price');
                
                // Update modal content
                document.querySelector('.payment-modal-title').textContent = `Purchase ${points} Points`;
                document.querySelector('.package-price-display').textContent = price;
                document.querySelector('#package-id').value = packageId;
                
                // Show modal
                pointsModal.style.display = 'flex';
            });
        });
        
        if (closeModal) {
            closeModal.addEventListener('click', () => {
                pointsModal.style.display = 'none';
            });
        }
        
        // Close modal on outside click
        window.addEventListener('click', (e) => {
            if (e.target === pointsModal) {
                pointsModal.style.display = 'none';
            }
        });
    }

    // Initialize all systems when DOM is loaded
    // Initialize market if on market page
    if (document.querySelector('.market-table')) {
        initializeMarket();
    }
    
    // Initialize gift system if on gifts page
    if (document.querySelector('.gift-items')) {
        initializeGiftSystem();
    }
    
    // Initialize points system if on points page
    if (document.querySelector('.points-packages')) {
        initializePointsSystem();
    }
});